﻿using System;
using System.Collections.Generic;
using System.Linq;
using Eto.Drawing;
using Eto.Forms;
using Rhino;

namespace SampleEtoTreeGrid
{
  class SampleEtoTreeGridDialog : Dialog<DialogResult>
  {
    private int m_expanded;
    private readonly CheckBox m_allow_collapsing;
    private readonly CheckBox m_allow_expanding;
    private readonly TreeGridView m_grid;
    private int m_new_item_count;
    private static readonly Image g_image = SampleIcons.SampleIcon.WithSize(16, 16);
    private Label m_hover_node_label;
    private bool m_cancel_label_edit;

    public SampleEtoTreeGridDialog()
    {
      Padding = new Padding(5);
      Resizable = false;
      Result = DialogResult.Cancel;
      Title = GetType().Name;
      WindowStyle = WindowStyle.Default;
      var layout = new DynamicLayout { DefaultSpacing = new Size(5, 5), Padding = new Padding(10) };
      m_grid = ImagesAndMenu();

      layout.AddSeparateRow(
        null,
        m_allow_expanding = new CheckBox { Text = "Allow Expanding", Checked = true },
        m_allow_collapsing = new CheckBox { Text = "Allow Collapsing", Checked = true },
        ShowHeaderCheckBox(m_grid),
        null
      );
      layout.AddSeparateRow(null, InsertButton(), AddChildButton(), RemoveButton(), ExpandButton(), CollapseButton(), null);
      layout.AddSeparateRow(null, EnabledCheck(), AllowMultipleSelect(), "Border", CreateBorderType(m_grid), null);
      layout.AddSeparateRow(null, ReloadDataButton(m_grid), null);

      layout.Add(m_grid, yscale: true);
      layout.Add(HoverNodeLabel());

      Content = layout;
    }

    Control ShowHeaderCheckBox(TreeGridView grid)
    {
      var control = new CheckBox { Text = "ShowHeader" };
      control.CheckedBinding.Bind(grid, g => g.ShowHeader);
      return control;
    }

    Control ReloadDataButton(TreeGridView grid)
    {
      var control = new Button { Text = "ReloadData" };
      control.Click += (sender, e) => grid.ReloadData();
      return control;
    }

    Control CreateBorderType(TreeGridView grid)
    {
      var borderType = new EnumDropDown<BorderType>();
      borderType.SelectedValueBinding.Bind(grid, g => g.Border);
      return borderType;
    }

    Control HoverNodeLabel()
    {
      m_hover_node_label = new Label();

      m_grid.MouseMove += (sender, e) =>
      {
        var cell = m_grid.GetCellAt(e.Location);
        if (cell != null)
          m_hover_node_label.Text = $"Item under mouse: {((TreeGridItem)cell.Item)?.Values[1] ?? "(no item)"}, Column: {cell.Column?.HeaderText ?? "(no column)"}";
      };

      return m_hover_node_label;
    }

    Control InsertButton()
    {
      var control = new Button { Text = "Insert" };
      control.Click += (sender, e) =>
      {
        var item = m_grid.SelectedItem as TreeGridItem;
        var parent = (item?.Parent ?? (ITreeGridItem)m_grid.DataStore) as TreeGridItem;
        if (parent != null)
        {
          var index = item != null ? parent.Children.IndexOf(item) : 0;
          parent.Children.Insert(index, CreateComplexTreeItem(0, "New Item " + m_new_item_count++, null));
          if (item != null)
            m_grid.ReloadItem(parent);
          else
            m_grid.ReloadData();
        }
      };
      return control;
    }

    Control AddChildButton()
    {
      var control = new Button { Text = "Add Child" };
      control.Click += (sender, e) =>
      {
        var item = m_grid.SelectedItem as TreeGridItem;
        if (item != null)
        {
          item.Children.Add(CreateComplexTreeItem(0, "New Item " + m_new_item_count++, null));
          m_grid.ReloadItem(item);
        }
      };
      return control;
    }

    Control RemoveButton()
    {
      var control = new Button { Text = "Remove" };
      control.Click += (sender, e) =>
      {
        if (m_grid.AllowMultipleSelection)
        {
          var parents = new List<ITreeGridItem>();
          bool reloadData = false;
          foreach (var item in m_grid.SelectedItems.OfType<ITreeGridItem>().ToList())
          {
            var parent = item.Parent as TreeGridItem;
            parent.Children.Remove(item);
            if (parent.Parent == null)
              reloadData = true;
            if (!parents.Contains(parent))
              parents.Add(parent);
          }
          if (reloadData)
            m_grid.ReloadData();
          else
          {
            foreach (var parent in parents)
            {
              m_grid.ReloadItem(parent);
            }
          }
        }
        else
        {
          var item = m_grid.SelectedItem as TreeGridItem;
          if (item != null)
          {
            var parent = item.Parent as TreeGridItem;
            parent.Children.Remove(item);
            if (parent.Parent == null)
              m_grid.ReloadData();
            else
              m_grid.ReloadItem(parent);
          }
        }
      };
      return control;
    }

    Control ExpandButton()
    {
      var control = new Button { Text = "Expand" };
      control.Click += (sender, e) =>
      {
        var item = m_grid.SelectedItem;
        if (item != null)
        {
          item.Expanded = true;
          m_grid.ReloadItem(item);
        }
      };
      return control;
    }

    Control CollapseButton()
    {
      var control = new Button { Text = "Collapse" };
      control.Click += (sender, e) =>
      {
        var item = m_grid.SelectedItem;
        if (item != null)
        {
          item.Expanded = false;
          m_grid.ReloadItem(item);
        }
      };
      return control;
    }

    Control CancelLabelEdit()
    {
      var control = new CheckBox { Text = "Cancel Edit" };
      control.CheckedChanged += (sender, e) => m_cancel_label_edit = control.Checked ?? false;
      return control;
    }

    Control EnabledCheck()
    {
      var control = new CheckBox { Text = "Enabled", Checked = m_grid.Enabled };
      control.CheckedChanged += (sender, e) => m_grid.Enabled = control.Checked ?? false;
      return control;
    }

    Control AllowMultipleSelect()
    {
      var control = new CheckBox { Text = "AllowMultipleSelection" };
      control.CheckedBinding.Bind(m_grid, t => t.AllowMultipleSelection);
      return control;
    }

    TreeGridItem CreateComplexTreeItem(int level, string name, Image image)
    {
      var item = new TreeGridItem
      {
        Expanded = m_expanded++ % 2 == 0
      };
      item.Values = new object[] { image, "col 0 - " + name, "col 1 - " + name };
      if (level < 4)
      {
        for (int i = 0; i < 4; i++)
        {
          item.Children.Add(CreateComplexTreeItem(level + 1, name + " " + i, image));
        }
      }
      return item;
    }

    TreeGridView ImagesAndMenu()
    {
      var control = new TreeGridView
      {
        Size = new Size(100, 200)
      };

      control.Columns.Add(new GridColumn
      {
        DataCell = new ImageTextCell(0, 1),
        HeaderText = "Image and Text",
        AutoSize = true,
        Resizable = true,
        Editable = false
      });

      control.Columns.Add(new GridColumn
      {
        DataCell = new TextBoxCell(2),
        HeaderText = "Text",
        AutoSize = true,
        Width = 150,
        Resizable = true,
        Editable = false
      });

      if (Platform.Supports<ContextMenu>())
      {
        var menu = new ContextMenu();
        var item = new ButtonMenuItem { Text = "Click Me!" };
        item.Click += delegate
        {
          if (m_grid.SelectedItems.Any())
            RhinoApp.WriteLine("Click, Items: {0}, Rows: {1}", SelectedItemsString(m_grid), SelectedRowsString(m_grid));
          else
            RhinoApp.WriteLine("Click, no item selected");
        };
        menu.Items.Add(item);

        control.ContextMenu = menu;
      }

      control.DataStore = CreateComplexTreeItem(0, "", g_image);
      LogEvents(control);
      return control;
    }

    static string SelectedRowsString(TreeGridView grid)
    {
      return string.Join(",", grid.SelectedRows.Select(r => r.ToString()).OrderBy(r => r));
    }

    static string SelectedItemsString(TreeGridView grid)
    {
      return string.Join(",", grid.SelectedItems.Cast<ITreeGridItem>().Select(GetDescription).OrderBy(r => r));
    }

    static string GetDescription(ITreeGridItem item)
    {
      var treeItem = item as TreeGridItem;
      if (treeItem != null)
        return Convert.ToString(treeItem.Values[1]);
      //return Convert.ToString(string.Join(", ", treeItem.Values.Select(r => Convert.ToString(r))));
      return Convert.ToString(item);
    }

    void LogEvents(TreeGridView control)
    {
      control.Activated += (sender, e) =>
      {
        RhinoApp.WriteLine("Activated, Item: {0}", GetDescription(e.Item));
      };
      control.SelectionChanged += delegate
      {
        RhinoApp.WriteLine("SelectionChanged, Rows: {0}", string.Join(", ", control.SelectedRows.Select(r => r.ToString())));
        RhinoApp.WriteLine("\t Items: {0}", string.Join(", ", control.SelectedItems.OfType<TreeGridItem>().Select(r => GetDescription(r))));
      };
      control.SelectedItemChanged += delegate
      {
        RhinoApp.WriteLine("SelectedItemChanged, Item: {0}", control.SelectedItem != null ? GetDescription(control.SelectedItem) : "<none selected>");
      };

      control.Expanding += (sender, e) =>
      {
        RhinoApp.WriteLine("Expanding, Item: {0}", GetDescription(e.Item));
        e.Cancel = !(m_allow_expanding.Checked ?? true);
      };
      control.Expanded += (sender, e) =>
      {
        RhinoApp.WriteLine("Expanded, Item: {0}", GetDescription(e.Item));
      };
      control.Collapsing += (sender, e) =>
      {
        RhinoApp.WriteLine("Collapsing, Item: {0}", GetDescription(e.Item));
        e.Cancel = !(m_allow_collapsing.Checked ?? true);
      };
      control.Collapsed += (sender, e) =>
      {
        RhinoApp.WriteLine("Collapsed, Item: {0}", GetDescription(e.Item));
      };
      control.ColumnHeaderClick += delegate (object sender, GridColumnEventArgs e)
      {
        RhinoApp.WriteLine("ColumnHeaderClick: {0}", e.Column);
      };

      control.CellClick += (sender, e) =>
      {
        RhinoApp.WriteLine($"CellClick, Row: {e.Row}, Column: {e.Column}, Item: {e.Item}, ColInfo: {e.GridColumn}");
      };

      control.CellDoubleClick += (sender, e) =>
      {
        RhinoApp.WriteLine($"CellDoubleClick, Row: {e.Row}, Column: {e.Column}, Item: {e.Item}, ColInfo: {e.GridColumn}");
      };

      control.MouseDown += (sender, e) =>
      {
        var cell = control.GetCellAt(e.Location);
        RhinoApp.WriteLine($"MouseDown, Cell Column: {cell.Column?.HeaderText}, Item: {GetDescription(cell.Item as ITreeGridItem)}");
      };

      control.MouseUp += (sender, e) =>
      {
        var cell = control.GetCellAt(e.Location);
        RhinoApp.WriteLine($"MouseUp, Cell Column: {cell.Column?.HeaderText}, Item: {GetDescription(cell.Item as ITreeGridItem)}");
      };

      control.MouseDoubleClick += (sender, e) =>
      {
        var cell = control.GetCellAt(e.Location);
        RhinoApp.WriteLine($"MouseDoubleClick, Cell Column: {cell.Column?.HeaderText}, Item: {GetDescription(cell.Item as ITreeGridItem)}");
      };
    }
  }
}
